/*
 * Decompiled with CFR 0.152.
 */
package org.hipparchus.linear;

import org.hipparchus.exception.LocalizedCoreFormats;
import org.hipparchus.exception.MathIllegalArgumentException;
import org.hipparchus.exception.MathIllegalStateException;
import org.hipparchus.exception.MathRuntimeException;
import org.hipparchus.linear.Array2DRowRealMatrix;
import org.hipparchus.linear.ArrayRealVector;
import org.hipparchus.linear.DecompositionSolver;
import org.hipparchus.linear.DiagonalMatrix;
import org.hipparchus.linear.MatrixUtils;
import org.hipparchus.linear.RealMatrix;
import org.hipparchus.linear.RealVector;
import org.hipparchus.linear.TriDiagonalTransformer;
import org.hipparchus.util.FastMath;
import org.hipparchus.util.Precision;

public class EigenDecompositionSymmetric {
    public static final double DEFAULT_EPSILON = 1.0E-12;
    private static final byte MAX_ITER = 30;
    private final double epsilon;
    private double[] eigenvalues;
    private ArrayRealVector[] eigenvectors;
    private RealMatrix cachedV;
    private DiagonalMatrix cachedD;
    private RealMatrix cachedVt;

    public EigenDecompositionSymmetric(RealMatrix matrix) {
        this(matrix, 1.0E-12, true);
    }

    public EigenDecompositionSymmetric(RealMatrix matrix, double epsilon, boolean decreasing) throws MathRuntimeException {
        this.epsilon = epsilon;
        MatrixUtils.checkSymmetric(matrix, epsilon);
        TriDiagonalTransformer transformer = new TriDiagonalTransformer(matrix);
        this.findEigenVectors(transformer.getMainDiagonalRef(), transformer.getSecondaryDiagonalRef(), transformer.getQ().getData(), decreasing);
    }

    public EigenDecompositionSymmetric(double[] main, double[] secondary) {
        this(main, secondary, 1.0E-12, true);
    }

    public EigenDecompositionSymmetric(double[] main, double[] secondary, double epsilon, boolean decreasing) {
        this.epsilon = epsilon;
        int size = main.length;
        double[][] z = new double[size][size];
        for (int i = 0; i < size; ++i) {
            z[i][i] = 1.0;
        }
        this.findEigenVectors((double[])main.clone(), (double[])secondary.clone(), z, decreasing);
    }

    public RealMatrix getV() {
        if (this.cachedV == null) {
            int m = this.eigenvectors.length;
            this.cachedV = MatrixUtils.createRealMatrix(m, m);
            for (int k = 0; k < m; ++k) {
                this.cachedV.setColumnVector(k, this.eigenvectors[k]);
            }
        }
        return this.cachedV;
    }

    public DiagonalMatrix getD() {
        if (this.cachedD == null) {
            this.cachedD = new DiagonalMatrix(this.eigenvalues);
        }
        return this.cachedD;
    }

    public double getEpsilon() {
        return this.epsilon;
    }

    public RealMatrix getVT() {
        if (this.cachedVt == null) {
            int m = this.eigenvectors.length;
            this.cachedVt = MatrixUtils.createRealMatrix(m, m);
            for (int k = 0; k < m; ++k) {
                this.cachedVt.setRowVector(k, this.eigenvectors[k]);
            }
        }
        return this.cachedVt;
    }

    public double[] getEigenvalues() {
        return (double[])this.eigenvalues.clone();
    }

    public double getEigenvalue(int i) {
        return this.eigenvalues[i];
    }

    public RealVector getEigenvector(int i) {
        return this.eigenvectors[i].copy();
    }

    public double getDeterminant() {
        double determinant = 1.0;
        for (double eigenvalue : this.eigenvalues) {
            determinant *= eigenvalue;
        }
        return determinant;
    }

    public RealMatrix getSquareRoot() {
        double[] sqrtEigenValues = new double[this.eigenvalues.length];
        for (int i = 0; i < this.eigenvalues.length; ++i) {
            double eigen = this.eigenvalues[i];
            if (eigen <= 0.0) {
                throw new MathRuntimeException(LocalizedCoreFormats.UNSUPPORTED_OPERATION, new Object[0]);
            }
            sqrtEigenValues[i] = FastMath.sqrt(eigen);
        }
        RealMatrix sqrtEigen = MatrixUtils.createRealDiagonalMatrix(sqrtEigenValues);
        RealMatrix v = this.getV();
        RealMatrix vT = this.getVT();
        return v.multiply(sqrtEigen).multiply(vT);
    }

    public DecompositionSolver getSolver() {
        return new Solver();
    }

    private void findEigenVectors(double[] main, double[] secondary, double[][] householderMatrix, boolean decreasing) {
        int i;
        double[][] z = (double[][])householderMatrix.clone();
        int n = main.length;
        this.eigenvalues = new double[n];
        double[] e = new double[n];
        for (int i2 = 0; i2 < n - 1; ++i2) {
            this.eigenvalues[i2] = main[i2];
            e[i2] = secondary[i2];
        }
        this.eigenvalues[n - 1] = main[n - 1];
        e[n - 1] = 0.0;
        double maxAbsoluteValue = 0.0;
        for (i = 0; i < n; ++i) {
            if (FastMath.abs(this.eigenvalues[i]) > maxAbsoluteValue) {
                maxAbsoluteValue = FastMath.abs(this.eigenvalues[i]);
            }
            if (!(FastMath.abs(e[i]) > maxAbsoluteValue)) continue;
            maxAbsoluteValue = FastMath.abs(e[i]);
        }
        if (maxAbsoluteValue != 0.0) {
            for (i = 0; i < n; ++i) {
                if (FastMath.abs(this.eigenvalues[i]) <= Precision.EPSILON * maxAbsoluteValue) {
                    this.eigenvalues[i] = 0.0;
                }
                if (!(FastMath.abs(e[i]) <= Precision.EPSILON * maxAbsoluteValue)) continue;
                e[i] = 0.0;
            }
        }
        for (int j = 0; j < n; ++j) {
            int m;
            int its = 0;
            do {
                int i3;
                for (m = j; m < n - 1; ++m) {
                    double delta = FastMath.abs(this.eigenvalues[m]) + FastMath.abs(this.eigenvalues[m + 1]);
                    if (FastMath.abs(e[m]) + delta == delta) break;
                }
                if (m == j) continue;
                if (its == 30) {
                    throw new MathIllegalStateException(LocalizedCoreFormats.CONVERGENCE_FAILED, (byte)30);
                }
                ++its;
                double q = (this.eigenvalues[j + 1] - this.eigenvalues[j]) / (2.0 * e[j]);
                double t = FastMath.sqrt(1.0 + q * q);
                q = q < 0.0 ? this.eigenvalues[m] - this.eigenvalues[j] + e[j] / (q - t) : this.eigenvalues[m] - this.eigenvalues[j] + e[j] / (q + t);
                double u = 0.0;
                double s = 1.0;
                double c = 1.0;
                for (i3 = m - 1; i3 >= j; --i3) {
                    double p = s * e[i3];
                    double h = c * e[i3];
                    if (FastMath.abs(p) >= FastMath.abs(q)) {
                        c = q / p;
                        t = FastMath.sqrt(c * c + 1.0);
                        e[i3 + 1] = p * t;
                        s = 1.0 / t;
                        c *= s;
                    } else {
                        s = p / q;
                        t = FastMath.sqrt(s * s + 1.0);
                        e[i3 + 1] = q * t;
                        c = 1.0 / t;
                        s *= c;
                    }
                    if (e[i3 + 1] == 0.0) {
                        int n2 = i3 + 1;
                        this.eigenvalues[n2] = this.eigenvalues[n2] - u;
                        e[m] = 0.0;
                        break;
                    }
                    q = this.eigenvalues[i3 + 1] - u;
                    t = (this.eigenvalues[i3] - q) * s + 2.0 * c * h;
                    u = s * t;
                    this.eigenvalues[i3 + 1] = q + u;
                    q = c * t - h;
                    for (int ia = 0; ia < n; ++ia) {
                        p = z[ia][i3 + 1];
                        z[ia][i3 + 1] = s * z[ia][i3] + c * p;
                        z[ia][i3] = c * z[ia][i3] - s * p;
                    }
                }
                if (t == 0.0 && i3 >= j) continue;
                int n3 = j;
                this.eigenvalues[n3] = this.eigenvalues[n3] - u;
                e[j] = q;
                e[m] = 0.0;
            } while (m != j);
        }
        for (i = 0; i < n; ++i) {
            int j;
            int k = i;
            double p = this.eigenvalues[i];
            for (j = i + 1; j < n; ++j) {
                if (this.eigenvalues[j] > p != decreasing) continue;
                k = j;
                p = this.eigenvalues[j];
            }
            if (k == i) continue;
            this.eigenvalues[k] = this.eigenvalues[i];
            this.eigenvalues[i] = p;
            for (j = 0; j < n; ++j) {
                p = z[j][i];
                z[j][i] = z[j][k];
                z[j][k] = p;
            }
        }
        maxAbsoluteValue = 0.0;
        for (i = 0; i < n; ++i) {
            if (!(FastMath.abs(this.eigenvalues[i]) > maxAbsoluteValue)) continue;
            maxAbsoluteValue = FastMath.abs(this.eigenvalues[i]);
        }
        if (maxAbsoluteValue != 0.0) {
            for (i = 0; i < n; ++i) {
                if (!(FastMath.abs(this.eigenvalues[i]) < Precision.EPSILON * maxAbsoluteValue)) continue;
                this.eigenvalues[i] = 0.0;
            }
        }
        this.eigenvectors = new ArrayRealVector[n];
        for (i = 0; i < n; ++i) {
            this.eigenvectors[i] = new ArrayRealVector(n);
            for (int j = 0; j < n; ++j) {
                this.eigenvectors[i].setEntry(j, z[j][i]);
            }
        }
    }

    private class Solver
    implements DecompositionSolver {
        private Solver() {
        }

        @Override
        public RealVector solve(RealVector b) {
            if (!this.isNonSingular()) {
                throw new MathIllegalArgumentException(LocalizedCoreFormats.SINGULAR_MATRIX, new Object[0]);
            }
            int m = EigenDecompositionSymmetric.this.eigenvalues.length;
            if (b.getDimension() != m) {
                throw new MathIllegalArgumentException(LocalizedCoreFormats.DIMENSIONS_MISMATCH, b.getDimension(), m);
            }
            double[] bp = new double[m];
            for (int i = 0; i < m; ++i) {
                ArrayRealVector v = EigenDecompositionSymmetric.this.eigenvectors[i];
                double[] vData = v.getDataRef();
                double s = v.dotProduct(b) / EigenDecompositionSymmetric.this.eigenvalues[i];
                for (int j = 0; j < m; ++j) {
                    int n = j;
                    bp[n] = bp[n] + s * vData[j];
                }
            }
            return new ArrayRealVector(bp, false);
        }

        @Override
        public RealMatrix solve(RealMatrix b) {
            if (!this.isNonSingular()) {
                throw new MathIllegalArgumentException(LocalizedCoreFormats.SINGULAR_MATRIX, new Object[0]);
            }
            int m = EigenDecompositionSymmetric.this.eigenvalues.length;
            if (b.getRowDimension() != m) {
                throw new MathIllegalArgumentException(LocalizedCoreFormats.DIMENSIONS_MISMATCH, b.getRowDimension(), m);
            }
            int nColB = b.getColumnDimension();
            double[][] bp = new double[m][nColB];
            double[] tmpCol = new double[m];
            for (int k = 0; k < nColB; ++k) {
                int i;
                for (i = 0; i < m; ++i) {
                    tmpCol[i] = b.getEntry(i, k);
                    bp[i][k] = 0.0;
                }
                for (i = 0; i < m; ++i) {
                    int j;
                    ArrayRealVector v = EigenDecompositionSymmetric.this.eigenvectors[i];
                    double[] vData = v.getDataRef();
                    double s = 0.0;
                    for (j = 0; j < m; ++j) {
                        s += v.getEntry(j) * tmpCol[j];
                    }
                    s /= EigenDecompositionSymmetric.this.eigenvalues[i];
                    for (j = 0; j < m; ++j) {
                        double[] dArray = bp[j];
                        int n = k;
                        dArray[n] = dArray[n] + s * vData[j];
                    }
                }
            }
            return new Array2DRowRealMatrix(bp, false);
        }

        @Override
        public boolean isNonSingular() {
            double largestEigenvalueNorm = 0.0;
            for (double v : EigenDecompositionSymmetric.this.eigenvalues) {
                largestEigenvalueNorm = FastMath.max(largestEigenvalueNorm, FastMath.abs(v));
            }
            if (largestEigenvalueNorm == 0.0) {
                return false;
            }
            for (double eigenvalue : EigenDecompositionSymmetric.this.eigenvalues) {
                if (!Precision.equals(FastMath.abs(eigenvalue) / largestEigenvalueNorm, 0.0, EigenDecompositionSymmetric.this.epsilon)) continue;
                return false;
            }
            return true;
        }

        @Override
        public RealMatrix getInverse() {
            if (!this.isNonSingular()) {
                throw new MathIllegalArgumentException(LocalizedCoreFormats.SINGULAR_MATRIX, new Object[0]);
            }
            int m = EigenDecompositionSymmetric.this.eigenvalues.length;
            double[][] invData = new double[m][m];
            for (int i = 0; i < m; ++i) {
                double[] invI = invData[i];
                for (int j = 0; j < m; ++j) {
                    double invIJ = 0.0;
                    for (int k = 0; k < m; ++k) {
                        double[] vK = EigenDecompositionSymmetric.this.eigenvectors[k].getDataRef();
                        invIJ += vK[i] * vK[j] / EigenDecompositionSymmetric.this.eigenvalues[k];
                    }
                    invI[j] = invIJ;
                }
            }
            return MatrixUtils.createRealMatrix(invData);
        }

        @Override
        public int getRowDimension() {
            return EigenDecompositionSymmetric.this.eigenvalues.length;
        }

        @Override
        public int getColumnDimension() {
            return EigenDecompositionSymmetric.this.eigenvalues.length;
        }
    }
}

