/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *      https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

/*
 * This is not the original file distributed by the Apache Software Foundation
 * It has been modified by the Hipparchus project
 */

package org.hipparchus.optim.nonlinear.scalar.gradient;

import java.util.ArrayList;

import org.hipparchus.analysis.MultivariateFunction;
import org.hipparchus.analysis.MultivariateVectorFunction;
import org.hipparchus.geometry.euclidean.twod.Vector2D;
import org.hipparchus.optim.nonlinear.scalar.ObjectiveFunction;
import org.hipparchus.optim.nonlinear.scalar.ObjectiveFunctionGradient;

/**
 * Class used in the tests.
 */
public class CircleScalar {
    private ArrayList<Vector2D> points;

    public CircleScalar() {
        points  = new ArrayList<Vector2D>();
    }

    public void addPoint(double px, double py) {
        points.add(new Vector2D(px, py));
    }

    public double getRadius(Vector2D center) {
        double r = 0;
        for (Vector2D point : points) {
            r += point.distance(center);
        }
        return r / points.size();
    }

    public ObjectiveFunction getObjectiveFunction() {
        return new ObjectiveFunction(new MultivariateFunction() {
                public double value(double[] params)  {
                    Vector2D center = new Vector2D(params[0], params[1]);
                    double radius = getRadius(center);
                    double sum = 0;
                    for (Vector2D point : points) {
                        double di = point.distance(center) - radius;
                        sum += di * di;
                    }
                    return sum;
                }
            });
    }

    public ObjectiveFunctionGradient getObjectiveFunctionGradient() {
        return new ObjectiveFunctionGradient(new MultivariateVectorFunction() {
                public double[] value(double[] params) {
                    Vector2D center = new Vector2D(params[0], params[1]);
                    double radius = getRadius(center);
                    // gradient of the sum of squared residuals
                    double dJdX = 0;
                    double dJdY = 0;
                    for (Vector2D pk : points) {
                        double dk = pk.distance(center);
                        dJdX += (center.getX() - pk.getX()) * (dk - radius) / dk;
                        dJdY += (center.getY() - pk.getY()) * (dk - radius) / dk;
                    }
                    dJdX *= 2;
                    dJdY *= 2;

                    return new double[] { dJdX, dJdY };
                }
            });
    }
}
