/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *      https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

/*
 * This is not the original file distributed by the Apache Software Foundation
 * It has been modified by the Hipparchus project
 */

package org.hipparchus.optim.nonlinear.scalar.noderiv;

import org.hipparchus.analysis.MultivariateFunction;
import org.hipparchus.analysis.MultivariateVectorFunction;
import org.hipparchus.exception.MathIllegalStateException;
import org.hipparchus.exception.MathRuntimeException;
import org.hipparchus.linear.Array2DRowRealMatrix;
import org.hipparchus.linear.RealMatrix;
import org.hipparchus.optim.InitialGuess;
import org.hipparchus.optim.MaxEval;
import org.hipparchus.optim.PointValuePair;
import org.hipparchus.optim.SimpleBounds;
import org.hipparchus.optim.nonlinear.scalar.GoalType;
import org.hipparchus.optim.nonlinear.scalar.LeastSquaresConverter;
import org.hipparchus.optim.nonlinear.scalar.ObjectiveFunction;
import org.hipparchus.util.FastMath;
import org.junit.jupiter.api.Test;

import static org.junit.jupiter.api.Assertions.assertEquals;
import static org.junit.jupiter.api.Assertions.assertThrows;
import static org.junit.jupiter.api.Assertions.assertTrue;

class SimplexOptimizerNelderMeadTest {
    @Test
    void testBoundsUnsupported() {
        assertThrows(MathRuntimeException.class, () -> {
            SimplexOptimizer optimizer = new SimplexOptimizer(1e-10, 1e-30);
            final FourExtrema fourExtrema = new FourExtrema();

            optimizer.optimize(new MaxEval(100),
                new ObjectiveFunction(fourExtrema),
                GoalType.MINIMIZE,
                new InitialGuess(new double[]{-3, 0}),
                new NelderMeadSimplex(new double[]{0.2, 0.2}),
                new SimpleBounds(new double[]{-5, -1},
                    new double[]{5, 1}));
        });
    }

    @Test
    void testMinimize1() {
        SimplexOptimizer optimizer = new SimplexOptimizer(1e-10, 1e-30);
        final FourExtrema fourExtrema = new FourExtrema();

        final PointValuePair optimum
            = optimizer.optimize(new MaxEval(100),
                                 new ObjectiveFunction(fourExtrema),
                                 GoalType.MINIMIZE,
                                 new InitialGuess(new double[] { -3, 0 }),
                                 new NelderMeadSimplex(new double[] { 0.2, 0.2 }));
        assertEquals(fourExtrema.xM, optimum.getPoint()[0], 2e-7);
        assertEquals(fourExtrema.yP, optimum.getPoint()[1], 2e-5);
        assertEquals(fourExtrema.valueXmYp, optimum.getValue(), 6e-12);
        assertTrue(optimizer.getEvaluations() > 60);
        assertTrue(optimizer.getEvaluations() < 90);

        // Check that the number of iterations is updated (MATH-949).
        assertTrue(optimizer.getIterations() > 0);
    }

    @Test
    void testMinimize2() {
        SimplexOptimizer optimizer = new SimplexOptimizer(1e-10, 1e-30);
        final FourExtrema fourExtrema = new FourExtrema();

        final PointValuePair optimum
            = optimizer.optimize(new MaxEval(100),
                                 new ObjectiveFunction(fourExtrema),
                                 GoalType.MINIMIZE,
                                 new InitialGuess(new double[] { 1, 0 }),
                                 new NelderMeadSimplex(new double[] { 0.2, 0.2 }));
        assertEquals(fourExtrema.xP, optimum.getPoint()[0], 5e-6);
        assertEquals(fourExtrema.yM, optimum.getPoint()[1], 6e-6);
        assertEquals(fourExtrema.valueXpYm, optimum.getValue(), 1e-11);
        assertTrue(optimizer.getEvaluations() > 60);
        assertTrue(optimizer.getEvaluations() < 90);

        // Check that the number of iterations is updated (MATH-949).
        assertTrue(optimizer.getIterations() > 0);
    }

    @Test
    void testMaximize1() {
        SimplexOptimizer optimizer = new SimplexOptimizer(1e-10, 1e-30);
        final FourExtrema fourExtrema = new FourExtrema();

        final PointValuePair optimum
            = optimizer.optimize(new MaxEval(100),
                                 new ObjectiveFunction(fourExtrema),
                                 GoalType.MAXIMIZE,
                                 new InitialGuess(new double[] { -3, 0 }),
                                 new NelderMeadSimplex(new double[] { 0.2, 0.2 }));
        assertEquals(fourExtrema.xM, optimum.getPoint()[0], 1e-5);
        assertEquals(fourExtrema.yM, optimum.getPoint()[1], 3e-6);
        assertEquals(fourExtrema.valueXmYm, optimum.getValue(), 3e-12);
        assertTrue(optimizer.getEvaluations() > 60);
        assertTrue(optimizer.getEvaluations() < 90);

        // Check that the number of iterations is updated (MATH-949).
        assertTrue(optimizer.getIterations() > 0);
    }

    @Test
    void testMaximize2() {
        SimplexOptimizer optimizer = new SimplexOptimizer(1e-10, 1e-30);
        final FourExtrema fourExtrema = new FourExtrema();

        final PointValuePair optimum
            = optimizer.optimize(new MaxEval(100),
                                 new ObjectiveFunction(fourExtrema),
                                 GoalType.MAXIMIZE,
                                 new InitialGuess(new double[] { 1, 0 }),
                                 new NelderMeadSimplex(new double[] { 0.2, 0.2 }));
        assertEquals(fourExtrema.xP, optimum.getPoint()[0], 4e-6);
        assertEquals(fourExtrema.yP, optimum.getPoint()[1], 5e-6);
        assertEquals(fourExtrema.valueXpYp, optimum.getValue(), 7e-12);
        assertTrue(optimizer.getEvaluations() > 60);
        assertTrue(optimizer.getEvaluations() < 90);

        // Check that the number of iterations is updated (MATH-949).
        assertTrue(optimizer.getIterations() > 0);
    }

    @Test
    void testRosenbrock() {

        Rosenbrock rosenbrock = new Rosenbrock();
        SimplexOptimizer optimizer = new SimplexOptimizer(-1, 1e-3);
        PointValuePair optimum
        = optimizer.optimize(new MaxEval(100),
                             new ObjectiveFunction(rosenbrock),
                             GoalType.MINIMIZE,
                             new InitialGuess(new double[] { -1.2, 1 }),
                                new NelderMeadSimplex(new double[][] {
                                        { -1.2,  1 },
                                        { 0.9, 1.2 },
                                        {  3.5, -2.3 } }));

        assertEquals(rosenbrock.getCount(), optimizer.getEvaluations());
        assertTrue(optimizer.getEvaluations() > 40);
        assertTrue(optimizer.getEvaluations() < 50);
        assertTrue(optimum.getValue() < 8e-4);
    }

    @Test
    void testPowell() {
        Powell powell = new Powell();
        SimplexOptimizer optimizer = new SimplexOptimizer(-1, 1e-3);
        PointValuePair optimum =
            optimizer.optimize(new MaxEval(200),
                               new ObjectiveFunction(powell),
                               GoalType.MINIMIZE,
                               new InitialGuess(new double[] { 3, -1, 0, 1 }),
                               new NelderMeadSimplex(4));
        assertEquals(powell.getCount(), optimizer.getEvaluations());
        assertTrue(optimizer.getEvaluations() > 110);
        assertTrue(optimizer.getEvaluations() < 130);
        assertTrue(optimum.getValue() < 2e-3);
    }

    @Test
    void testLeastSquares1() {
        final RealMatrix factors
            = new Array2DRowRealMatrix(new double[][] {
                    { 1, 0 },
                    { 0, 1 }
                }, false);
        LeastSquaresConverter ls = new LeastSquaresConverter(new MultivariateVectorFunction() {
                public double[] value(double[] variables) {
                    return factors.operate(variables);
                }
            }, new double[] { 2.0, -3.0 });
        SimplexOptimizer optimizer = new SimplexOptimizer(-1, 1e-6);
        PointValuePair optimum =
            optimizer.optimize(new MaxEval(200),
                               new ObjectiveFunction(ls),
                               GoalType.MINIMIZE,
                               new InitialGuess(new double[] { 10, 10 }),
                               new NelderMeadSimplex(2));
        assertEquals( 2, optimum.getPointRef()[0], 3e-5);
        assertEquals(-3, optimum.getPointRef()[1], 4e-4);
        assertTrue(optimizer.getEvaluations() > 60);
        assertTrue(optimizer.getEvaluations() < 80);
        assertTrue(optimum.getValue() < 1.0e-6);
    }

    @Test
    void testLeastSquares2() {
        final RealMatrix factors
            = new Array2DRowRealMatrix(new double[][] {
                    { 1, 0 },
                    { 0, 1 }
                }, false);
        LeastSquaresConverter ls = new LeastSquaresConverter(new MultivariateVectorFunction() {
                public double[] value(double[] variables) {
                    return factors.operate(variables);
                }
            }, new double[] { 2, -3 }, new double[] { 10, 0.1 });
        SimplexOptimizer optimizer = new SimplexOptimizer(-1, 1e-6);
        PointValuePair optimum =
            optimizer.optimize(new MaxEval(200),
                               new ObjectiveFunction(ls),
                               GoalType.MINIMIZE,
                               new InitialGuess(new double[] { 10, 10 }),
                               new NelderMeadSimplex(2));
        assertEquals( 2, optimum.getPointRef()[0], 5e-5);
        assertEquals(-3, optimum.getPointRef()[1], 8e-4);
        assertTrue(optimizer.getEvaluations() > 60);
        assertTrue(optimizer.getEvaluations() < 80);
        assertTrue(optimum.getValue() < 1e-6);
    }

    @Test
    void testLeastSquares3() {
        final RealMatrix factors =
            new Array2DRowRealMatrix(new double[][] {
                    { 1, 0 },
                    { 0, 1 }
                }, false);
        LeastSquaresConverter ls = new LeastSquaresConverter(new MultivariateVectorFunction() {
                public double[] value(double[] variables) {
                    return factors.operate(variables);
                }
            }, new double[] { 2, -3 }, new Array2DRowRealMatrix(new double [][] {
                    { 1, 1.2 }, { 1.2, 2 }
                }));
        SimplexOptimizer optimizer = new SimplexOptimizer(-1, 1e-6);
        PointValuePair optimum
            = optimizer.optimize(new MaxEval(200),
                                 new ObjectiveFunction(ls),
                                 GoalType.MINIMIZE,
                                 new InitialGuess(new double[] { 10, 10 }),
                                 new NelderMeadSimplex(2));
        assertEquals( 2, optimum.getPointRef()[0], 2e-3);
        assertEquals(-3, optimum.getPointRef()[1], 8e-4);
        assertTrue(optimizer.getEvaluations() > 60);
        assertTrue(optimizer.getEvaluations() < 80);
        assertTrue(optimum.getValue() < 1e-6);
    }

    @Test
    void testMaxIterations() {
        assertThrows(MathIllegalStateException.class, () -> {
            Powell powell = new Powell();
            SimplexOptimizer optimizer = new SimplexOptimizer(-1, 1e-3);
            optimizer.optimize(new MaxEval(20),
                new ObjectiveFunction(powell),
                GoalType.MINIMIZE,
                new InitialGuess(new double[]{3, -1, 0, 1}),
                new NelderMeadSimplex(4));
        });
    }

    private static class FourExtrema implements MultivariateFunction {
        // The following function has 4 local extrema.
        final double xM = -3.841947088256863675365;
        final double yM = -1.391745200270734924416;
        final double xP =  0.2286682237349059125691;
        final double yP = -yM;
        final double valueXmYm = 0.2373295333134216789769; // Local maximum.
        final double valueXmYp = -valueXmYm; // Local minimum.
        final double valueXpYm = -0.7290400707055187115322; // Global minimum.
        final double valueXpYp = -valueXpYm; // Global maximum.

        public double value(double[] variables) {
            final double x = variables[0];
            final double y = variables[1];
            return (x == 0 || y == 0) ? 0 :
                FastMath.atan(x) * FastMath.atan(x + 2) * FastMath.atan(y) * FastMath.atan(y) / (x * y);
        }
    }

    private static class Rosenbrock implements MultivariateFunction {
        private int count;

        public Rosenbrock() {
            count = 0;
        }

        public double value(double[] x) {
            ++count;
            double a = x[1] - x[0] * x[0];
            double b = 1.0 - x[0];
            return 100 * a * a + b * b;
        }

        public int getCount() {
            return count;
        }
    }

    private static class Powell implements MultivariateFunction {
        private int count;

        public Powell() {
            count = 0;
        }

        public double value(double[] x) {
            ++count;
            double a = x[0] + 10 * x[1];
            double b = x[2] - x[3];
            double c = x[1] - 2 * x[2];
            double d = x[0] - x[3];
            return a * a + 5 * b * b + c * c * c * c + 10 * d * d * d * d;
        }

        public int getCount() {
            return count;
        }
    }
}
